<?php
/* --------------------------------------------------------------
   WithdrawalFilter.php 2020-04-16
   Gambio GmbH
   http://www.gambio.de
   Copyright (c) 2020 Gambio GmbH
   Released under the GNU General Public License (Version 2)
   [http://www.gnu.org/licenses/gpl-2.0.html]
   --------------------------------------------------------------
*/

declare(strict_types=1);

namespace Gambio\Admin\Withdrawal\SqlCriteria;

use Gambio\Core\Criteria\SqlFilter;
use Webmozart\Assert\Assert;

/**
 * Class WithdrawalFilter
 *
 * @package Gambio\Admin\Withdrawal\SqlCriteria
 */
class WithdrawalFilter implements SqlFilter
{
    /**
     * @var string
     */
    private $attribute;
    
    /**
     * @var string
     */
    private $value;
    
    /**
     * @var string
     */
    private $operation;
    
    /**
     * @var string[]
     */
    private static $attributeColumnMapping = [
        'id'                        => 'withdrawal_id',
        'order.id'                  => 'order_id',
        'order.creationDate'        => 'order_date',
        'order.deliveryDate'        => 'delivery_date',
        'customer.id'               => 'customer_id',
        'customer.gender'           => 'customer_gender',
        'customer.firstName'        => 'customer_firstname',
        'customer.lastName'         => 'customer_lastname',
        'customer.address.street'   => 'customer_street_address',
        'customer.address.postcode' => 'customer_postcode',
        'customer.address.city'     => 'customer_city',
        'customer.address.country'  => 'customer_country',
        'customer.email'            => 'customer_email',
        'date'                      => 'withdrawal_date',
        'content'                   => 'withdrawal_content',
        'createdByAdmin'            => 'created_by_admin',
        'createdOn'                 => 'date_created',
    ];
    
    /**
     * @var string[]
     */
    private static $allowedForLikeOperation = [
        'customer.gender',
        'customer.firstName',
        'customer.lastName',
        'customer.address.street',
        'customer.address.postcode',
        'customer.address.city',
        'customer.address.country',
        'customer.email',
        'content',
    ];
    
    /**
     * @var string[]
     */
    private static $allowedForNumericOperation = [
        'id',
        'order.id',
        'order.creationDate',
        'order.deliveryDate',
        'customer.id',
        'date',
        'createdOn',
    ];
    
    
    /**
     * WithdrawalFilter constructor.
     *
     * @param string $attribute
     * @param string $value
     * @param string $operation
     */
    public function __construct(string $attribute, string $value, string $operation)
    {
        $this->attribute = $attribute;
        $this->value     = $value;
        $this->operation = $operation;
    }
    
    
    /**
     * @param string $attribute
     * @param string $value
     * @param string $operation
     *
     * @return WithdrawalFilter
     */
    public static function create(string $attribute, string $value, string $operation = 'like'): WithdrawalFilter
    {
        $attributes = array_keys(self::$attributeColumnMapping);
        Assert::oneOf($attribute,
                      $attributes,
                      'Provided attribute needs to be one of: ' . implode(', ', $attributes) . '; Got: ' . $attribute);
        
        $operations = ['like', 'eq', 'gt', 'gte', 'lt', 'lte'];
        Assert::oneOf($operation,
                      $operations,
                      'Provided operation needs to be one of: ' . implode(', ', $operations) . '; Got: ' . $operation);
        
        if ($operation === 'like') {
            Assert::oneOf($attribute,
                          self::$allowedForLikeOperation,
                          'Provided attribute is not allowed for "like" operation.');
        } elseif (in_array($operation, ['gt', 'gte', 'lt', 'lte'])) {
            Assert::oneOf($attribute,
                          self::$allowedForNumericOperation,
                          'Provided attribute is not allowed for a numeric operation.');
        }
        
        return new self($attribute, $value, $operation);
    }
    
    
    /**
     * @return string
     */
    public function attribute(): string
    {
        return $this->attribute;
    }
    
    
    /**
     * @return string
     */
    public function value(): string
    {
        return $this->value;
    }
    
    
    /**
     * @return string
     */
    public function operation(): string
    {
        return $this->operation;
    }
    
    
    /**
     * @return string
     */
    public function column(): string
    {
        return self::$attributeColumnMapping[$this->attribute];
    }
}